/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	pgpDriver9x.c - main C routines for PGP utility VxD (Win95/98)
	

	$Id: pgpDriver9x.c,v 1.9 2001/04/06 13:34:18 pbj Exp $
____________________________________________________________________________*/


#define WANTVXDWRAPS
#include <basedef.h>
#include <vmm.h>
#include <vtd.h>
#include <vwin32.h>
#include <vxdwraps.h>

#include <debug.h>
#include <winerror.h>
#include <memory.h>

#ifndef PGP_EROS
#define PGP_EROS	0
#endif // PGP_EROS

#include "PGPsdkDriver.h"

#include "pgpDriver9x.h"
#include "pgpMemlock9x.h"
#include "pgpGeneric.h"
#include "pgpMisc.h"

#if !PGP_EROS
#include "pgpWipeDelete9x.h"
#include "pgpEntropy.h"
#endif // !PGP_EROS

// prototypes
void OnSystemExit (void);

// globals
PGPUTILCONTEXT	g_context;

#if !PGP_EROS
//____________________________________________________
// 
//	called by callback function to add entropy to pool

void 
PGPKBDEvent (WORD wScanCode)
{
	ULONG	ulForceAdd	= FALSE;

	g_context.ulStatusFlags |= kPGPUDFlag_KeyboardHookCalled;

	if (--g_context.iKBDCount < 0)
	{
		g_context.iKBDCount = KEYBOARD_SKIP;
		ulForceAdd = TRUE;
	}

	// leave as SPINLOCK Win95/98 kernel code is non-multitasking and it
	// is absolutely too dangerous to take a mutex at this time
	if (pgpDriverTryToEnterCriticalSection (&g_context.csEntropy, 
		SPINLOCK))
	{
		pgpEntropyAddKeystrokeEvent (ulForceAdd, wScanCode);
		pgpDriverLeaveCriticalSection (&g_context.csEntropy, SPINLOCK);
	}

	pgpProcessActivityEvent (&g_context.inactivity,
									&g_context.csInactivity);
}


//____________________________________________________
// 
//	called by callback function to add entropy to pool

void 
PGPMouseMoved (WORD wButtons)
{
	ULONG	ulForceAdd	= FALSE;

	g_context.ulStatusFlags |= kPGPUDFlag_MouseHookCalled;

	if (--g_context.iMouseCount < 0)
	{
		g_context.iMouseCount = MOUSE_SKIP;
		ulForceAdd = TRUE;
	}

	// leave as SPINLOCK Win95/98 kernel code is non-multitasking and it
	// is absolutely too dangerous to take a mutex at this time

	if (pgpDriverTryToEnterCriticalSection (&g_context.csEntropy, 
		SPINLOCK))
	{
		pgpEntropyAddMouseEvent (ulForceAdd);
		pgpDriverLeaveCriticalSection (&g_context.csEntropy, SPINLOCK);
	}

	// reset inactivity timer only if a button is down
	if (wButtons & 0x003C)
	{
		pgpProcessActivityEvent (&g_context.inactivity,
									&g_context.csInactivity);
	}
}


//	______________________________________________________
//
//	called by the system once per second (note we don't care
//  about values passed to us in registers, so we can 
//  just use a C function)

static VOID
sTimerCallback (VOID)
{
	g_context.ulStatusFlags |= kPGPUDFlag_InactivityTimerRunning;

	pgpProcessCacheTimerEvent (
			&g_context.cacheSign, &g_context.csCache);
	pgpProcessCacheTimerEvent (
			&g_context.cacheDecrypt, &g_context.csCache);
	pgpProcessInactivityTimerEvent (
			&g_context.inactivity, &g_context.csInactivity);

	// setup the next callback
	Set_Global_Time_Out (sTimerCallback, 1000, 0);
}


//	______________________________________________________
//
//  called upon when client VxD calls our Service API

BOOL
OnPGPutilInactivity (PPGPINACTIVITYSTRUCT pStruct)
{
	PGPdbgVerbosePrint (("PGPutil: OnPGPutilInactivity.\n"));

	pgpInactivityProcessOperation (
			&g_context.inactivity,
			pStruct, 
			&g_context.csInactivity);

	return TRUE;    	// success
}
#endif // !PGP_EROS


//	______________________________________________________
//
//  just make sure that the buffer is big enough to hold the data

static ULONG
sValidateDeviceIOBuffer (
    IN PDIOCPARAMETERS		p,
	IN ULONG				ulStructSize)
{
	if ((!p->lpvInBuffer) || (!p->lpvOutBuffer))
	{
		PGPdbgPrint (("PGPutil: OnDeviceIoControl: "
						"bad buffer(s):\n"));
		return FALSE;
	}

	if (p->cbInBuffer < ulStructSize) 
	{
		PGPdbgPrint (("PGPutil: OnDeviceIoControl: "
						"bad cbInBuffer: %X.\n",
						p->cbInBuffer));
		return FALSE;
	}

	if (p->cbOutBuffer < ulStructSize)
	{
		PGPdbgPrint (("PGPutil: OnDeviceIoControl: "
						"bad cbOutBuffer: %X.\n",
						p->cbOutBuffer));
		return FALSE;
	}

	return TRUE;
}


//	______________________________________________________
//
//  called upon as result of call to DeviceIoControl

DWORD 
OnDeviceIoControl (PDIOCPARAMETERS p) 
{
	DWORD	dwReturn	= ERROR_INVALID_PARAMETER;
	PVOID	pStruct		= NULL;

	pStruct = (PVOID)(p->lpvInBuffer);

    switch (p->dwIoControlCode) {
    case DIOC_GETVERSION :
        dwReturn = 0;	// no error
		break;

	case IOCTL_PGPUTIL_GENERIC :
		if (sValidateDeviceIOBuffer (p, sizeof(PPGPGENERICSTRUCT)))
		{
			pgpGenericProcessOperation (
					(PPGPGENERICSTRUCT)pStruct, g_context.ulStatusFlags);
			dwReturn = 0;
		}
		break;

#if !PGP_EROS
	case IOCTL_PGPUTIL_ENTROPY :
		if (sValidateDeviceIOBuffer (p, sizeof(PGPENTROPYSTRUCT)))
		{
			pgpEntropyProcessOperation (
					(PPGPENTROPYSTRUCT)pStruct, &g_context.csEntropy);
			dwReturn = 0;
		}
		break;

	case IOCTL_PGPUTIL_CACHE :
		if (sValidateDeviceIOBuffer (p, sizeof(PGPCACHESTRUCT)))
		{
			PPGPCACHE ppc	= NULL;

			switch (((PPGPCACHESTRUCT)pStruct)->ulCache) {
			case kPGPUDCache_Signing :
				ppc = &g_context.cacheSign;
				break;
			case kPGPUDCache_Decryption :
				ppc = &g_context.cacheDecrypt;
				break;
			default :
				((PPGPCACHESTRUCT)pStruct)->ulError = 
									kPGPUDError_BadParams;
				break;
			}

			if (ppc)
			{
				pgpCacheProcessOperation (
						ppc,
						(PPGPCACHESTRUCT)pStruct, 
						&g_context.csCache);
			}
			dwReturn = 0;
		}
		break;

	case IOCTL_PGPUTIL_WIPEDELETE :
		if (sValidateDeviceIOBuffer (p, sizeof(PGPWIPEDELETESTRUCT)))
		{
			pgpWipeDeleteProcessOperation (
					(PPGPWIPEDELETESTRUCT)pStruct,
					g_context.ulStatusFlags);
			dwReturn = 0;
		}
		break;
#endif // !PGP_EROS

	case IOCTL_PGPUTIL_MEMLOCK :
		if (sValidateDeviceIOBuffer (p, sizeof(PGPMEMLOCKSTRUCT)))
		{
			pgpMemlockProcessOperation (
					&g_context.memlock,
					(PPGPMEMLOCKSTRUCT)pStruct,
					p->hDevice,
					g_context.ulStatusFlags,
					&g_context.csMemlock);
			dwReturn = 0;
		}
		break;

	case DIOC_CLOSEHANDLE :
		pgpMemlockCleanupHandle (
				&g_context.memlock,
				p->hDevice,
				&g_context.csMemlock);
		dwReturn = 0;	// no error
		break;

    default :
		PGPdbgPrint (("PGPutil: Err: unknown IoControlCode: %X.\n",
						p->dwIoControlCode));
		dwReturn = ERROR_BAD_COMMAND;
    }

	return dwReturn;
}


//	______________________________________________________
//
//  called upon initialization of VxD

BOOL
OnDeviceInit ()
{
	PGPdbgVerbosePrint (("PGPutil: OnDeviceInit.\n"));

	// initialize context structure
	g_context.ulStatusFlags = 0;
	g_context.iKBDCount = 0;
	g_context.iMouseCount = 0;

#if !PGP_EROS
	// initialize the FS hook
	if (pgpWipeDeleteInit () == VXD_SUCCESS)
		g_context.ulStatusFlags |= kPGPUDFlag_WipeDeleteInitialized;
	else
		return VXD_FAILURE;

	// initialize critical section mutex
	pgpDriverInitCriticalSection (&(g_context.csCache));
	pgpDriverInitCriticalSection (&(g_context.csEntropy));
	pgpDriverInitCriticalSection (&(g_context.csInactivity));
#endif // !PGP_EROS
	pgpDriverInitCriticalSection (&(g_context.csMemlock));

	// initialize memlock heap
	pgpMemlockInit (&g_context.memlock);
	g_context.ulStatusFlags |= kPGPUDFlag_MemlockInitialized;

#if !PGP_EROS
	// initialize random pool
	pgpEntropyInit ();

	// initialize inactivity timer
	pgpInitInactivity (&(g_context.inactivity));

	// initialize caches
	pgpInitCache (&(g_context.cacheSign));
	pgpInitCache (&(g_context.cacheDecrypt));

	// initialize the 1-second callback timer
	Set_Global_Time_Out (sTimerCallback, 1000, 0);

	// assume no errors (!) on hooking mouse/keyboard
	g_context.ulStatusFlags |= kPGPUDFlag_KeyboardHookInstalled;
	g_context.ulStatusFlags |= kPGPUDFlag_MouseHookInstalled;
#endif // !PGP_EROS

	return VXD_SUCCESS;    	// success
}


//	______________________________________________________
//
//  called upon shutdown of VxD

BOOL
OnSystemExitC ()
{
	PGPdbgVerbosePrint (("PGPutil: OnSystemExit.\n"));

	// shutdown memlock code
	pgpMemlockCleanup (&g_context.memlock, &g_context.csMemlock);

#if !PGP_EROS
	// shutdown wipe-on-delete code
	pgpWipeDeleteCleanup ();

	// cleanup critical section mutexes
	pgpDriverCleanupCriticalSection (&(g_context.csCache));
	pgpDriverCleanupCriticalSection (&(g_context.csEntropy));
	pgpDriverCleanupCriticalSection (&(g_context.csInactivity));
#endif // !PGP_EROS

	pgpDriverCleanupCriticalSection (&(g_context.csMemlock));

#if !PGP_EROS
	// now call the assembly language routine to finish exiting
	OnSystemExit ();
#endif // !PGP_EROS

	return TRUE;    	// success
}


